// Licensed GNU LGPL v3 or later: http://www.gnu.org/licenses/lgpl.html

#include "smfft.hh"
#include "smutils.hh"
#include <algorithm>
#include <map>
#include "config.h"

#include <glib.h>
#include <string.h>
#include <unistd.h>

using namespace SpectMorph;
using std::map;
using std::string;

static bool enable_gsl_fft = false;
static bool randomize_new_fft_arrays = false;

void
FFT::use_gsl_fft (bool new_enable_gsl_fft)
{
  enable_gsl_fft = new_enable_gsl_fft;
}

void
FFT::debug_randomize_new_arrays (bool b)
{
  randomize_new_fft_arrays = b;
}

#if SPECTMORPH_HAVE_FFTW

#include <fftw3.h>

static void save_wisdom();

float *
FFT::new_array_float (size_t N)
{
  const size_t N_2 = N + 2; /* extra space for r2c extra complex output */

  float *result = (float *) fftwf_malloc (sizeof (float) * N_2);

  if (randomize_new_fft_arrays)
    {
      for (size_t i = 0; i < N_2; i++)
        result[i] = g_random_double_range (-1, 1);
    }
  return result;
}

void
FFT::free_array_float (float *f)
{
  fftwf_free (f);
}

static map<int, fftwf_plan> fftar_float_plan;

static int
plan_flags (FFT::PlanMode plan_mode)
{
  switch (plan_mode)
    {
    case FFT::PLAN_PATIENT:    return (FFTW_PATIENT | FFTW_PRESERVE_INPUT | FFTW_WISDOM_ONLY);
    case FFT::PLAN_ESTIMATE:   return (FFTW_ESTIMATE | FFTW_PRESERVE_INPUT);
    default:                   g_assert_not_reached();
    }
}

void
FFT::fftar_float (size_t N, float *in, float *out, PlanMode plan_mode)
{
  fftwf_plan& plan = fftar_float_plan[N];

  if (!plan)
    {
      float *plan_in = new_array_float (N);
      float *plan_out = new_array_float (N);
      plan = fftwf_plan_dft_r2c_1d (N, plan_in, (fftwf_complex *) plan_out, plan_flags (plan_mode));
      if (!plan) /* missing from wisdom -> create plan and save it */
        {
          plan = fftwf_plan_dft_r2c_1d (N, plan_in, (fftwf_complex *) plan_out, plan_flags (plan_mode) & ~FFTW_WISDOM_ONLY);
          save_wisdom();
        }
    }
  fftwf_execute_dft_r2c (plan, in, (fftwf_complex *) out);

  out[1] = out[N];
}

static map<int, fftwf_plan> fftsr_float_plan;

void
FFT::fftsr_float (size_t N, float *in, float *out, PlanMode plan_mode)
{
  fftwf_plan& plan = fftsr_float_plan[N];

  if (!plan)
    {
      float *plan_in = new_array_float (N);
      float *plan_out = new_array_float (N);
      plan = fftwf_plan_dft_c2r_1d (N, (fftwf_complex *) plan_in, plan_out, plan_flags (plan_mode));
      if (!plan) /* missing from wisdom -> create plan and save it */
        {
          plan = fftwf_plan_dft_c2r_1d (N, (fftwf_complex *) plan_in, plan_out, plan_flags (plan_mode) & ~FFTW_WISDOM_ONLY);
          save_wisdom();
        }
    }
  in[N] = in[1];
  in[N+1] = 0;
  in[1] = 0;

  fftwf_execute_dft_c2r (plan, (fftwf_complex *)in, out);

  in[1] = in[N]; // we need to preserve the input array
}

static map<int, fftwf_plan> fftsr_destructive_float_plan;

void
FFT::fftsr_destructive_float (size_t N, float *in, float *out, PlanMode plan_mode)
{
  fftwf_plan& plan = fftsr_destructive_float_plan[N];

  if (!plan)
    {
      int xplan_flags = plan_flags (plan_mode) & ~FFTW_PRESERVE_INPUT;
      float *plan_in = new_array_float (N);
      float *plan_out = new_array_float (N);
      plan = fftwf_plan_dft_c2r_1d (N, (fftwf_complex *) plan_in, plan_out, xplan_flags);
      if (!plan) /* missing from wisdom -> create plan and save it */
        {
          plan = fftwf_plan_dft_c2r_1d (N, (fftwf_complex *) plan_in, plan_out,
                                        xplan_flags & ~FFTW_WISDOM_ONLY);
          save_wisdom();
        }
    }
  in[N] = in[1];
  in[N+1] = 0;
  in[1] = 0;

  fftwf_execute_dft_c2r (plan, (fftwf_complex *)in, out);
}

static map<int, fftwf_plan> fftac_float_plan;

void
FFT::fftac_float (size_t N, float *in, float *out, PlanMode plan_mode)
{
  fftwf_plan& plan = fftac_float_plan[N];
  if (!plan)
    {
      float *plan_in = new_array_float (N * 2);
      float *plan_out = new_array_float (N * 2);

      plan = fftwf_plan_dft_1d (N, (fftwf_complex *) plan_in, (fftwf_complex *) plan_out,
                                FFTW_FORWARD, plan_flags (plan_mode));
      if (!plan) /* missing from wisdom -> create plan and save it */
        {
          plan = fftwf_plan_dft_1d (N, (fftwf_complex *) plan_in, (fftwf_complex *) plan_out,
                                    FFTW_FORWARD, plan_flags (plan_mode) & ~FFTW_WISDOM_ONLY);
          save_wisdom();
        }
    }

  fftwf_execute_dft (plan, (fftwf_complex *)in, (fftwf_complex *)out);
}

static map<int, fftwf_plan> fftsc_float_plan;

void
FFT::fftsc_float (size_t N, float *in, float *out, PlanMode plan_mode)
{
  fftwf_plan& plan = fftsc_float_plan[N];
  if (!plan)
    {
      float *plan_in = new_array_float (N * 2);
      float *plan_out = new_array_float (N * 2);

      plan = fftwf_plan_dft_1d (N, (fftwf_complex *) plan_in, (fftwf_complex *) plan_out,
                                FFTW_BACKWARD, plan_flags (plan_mode));
      if (!plan) /* missing from wisdom -> create plan and save it */
        {
          plan = fftwf_plan_dft_1d (N, (fftwf_complex *) plan_in, (fftwf_complex *) plan_out,
                                    FFTW_BACKWARD, plan_flags (plan_mode) & ~FFTW_WISDOM_ONLY);
          save_wisdom();
        }
     }
  fftwf_execute_dft (plan, (fftwf_complex *)in, (fftwf_complex *)out);
}

static string
wisdom_filename()
{
  const char *homedir = g_get_home_dir();
  const char *hostname = g_get_host_name();
  return homedir + string ("/.spectmorph_fftw_wisdom_") + hostname;
}

static void
save_wisdom()
{
  /* detect if we're running in valgrind - in this case newly accumulated wisdom is probably flawed */
  bool valgrind = false;

  FILE *maps = fopen (string_printf ("/proc/%d/maps", getpid()).c_str(), "r");
  if (maps)
    {
      char buffer[1024];
      while (fgets (buffer, 1024, maps))
        {
          if (strstr (buffer, "vgpreload"))
            valgrind = true;
        }
      fclose (maps);
    }
  if (valgrind)
    {
      printf ("FFT::save_wisdom(): not saving fft wisdom (running under valgrind)\n");
      return;
    }
  /* atomically replace old wisdom file with new wisdom file
   *
   * its theoretically possible (but highly unlikely) that we leak a *wisdom*.new.12345 file
   */
  string new_wisdom_filename = string_printf ("%s.new.%d", wisdom_filename().c_str(), getpid());
  FILE *outfile = fopen (new_wisdom_filename.c_str(), "w");
  if (outfile)
    {
      fftwf_export_wisdom_to_file (outfile);
      fclose (outfile);
      rename (new_wisdom_filename.c_str(), wisdom_filename().c_str());
    }
}

void
FFT::load_wisdom()
{
  FILE *infile = fopen (wisdom_filename().c_str(), "r");
  if (infile)
    {
      fftwf_import_wisdom_from_file (infile);
      fclose (infile);
    }
}

#else

#error "building without FFTW is not supported currently"

#endif
